
//Chris' WinGlide enhancements
//Copr. 1998, Chris Dohnal (cwdohnal@ucdavis.edu)

#include <windows.h>
#include <glide.h>
#include <ddraw.h>
#include "pointers.h"
#include "pcopy.h"
#include "main.h"
#include "dxdraw.h"
#include "errcode.h"
#include "msgbox.h"
#include "gamma.h"
#include "inifile.h"
#include "sysmenu.h"
#include "gammadlg.h"
#include "subclass.h"

BOOL OssWGInitialize(HWND, DWORD, DWORD);
VOID OssWGShutdown(VOID);
VOID OssWGCopy3DfxFrameBuffer(VOID);

static BOOL gbDisplayErrorMessage = FALSE;
static char gszErrorMessage[256] = "";
static HRESULT ghPreviousErrorCode = DD_OK;

static WNDPROC gpPrevWndProc = NULL;
static LRESULT CALLBACK SubclassWndProc(HWND, UINT, WPARAM, LPARAM);

static HBRUSH ghColorKeyBrush = NULL;

static HWND ghSubclassedWnd = NULL;

static VOID ConstructFlipOffScreenSurfaceFailedErrorMessage(HRESULT);
static VOID DisplayCouldNotFlipOffScreenSurfaceErrorMessage(HWND, HRESULT, BOOL);
static VOID DisplayCouldNotCreateOffScreenSurfaceErrorMessage(HRESULT);


BOOL OssWGInitialize(HWND hWnd, DWORD dwWidth, DWORD dwHeight) {
	LONG lRet;
	BOOL bRet;

	//Set defaults
	gbDisplayErrorMessage = FALSE;
	ghPreviousErrorCode = DD_OK;
	
	//Clear flags and pointers
	ghColorKeyBrush = NULL;

	
	//Initialize DirectDraw
	lRet = InitializeDirectDraw();
	if (lRet != EC_OK) {
		WinGlideErrorMessageBox(NULL, "Could not initialize DirectDraw.");

		goto initialization_failure;
	}

	//Set DirectDraw cooperative level
	lRet = SetNormalCooperativeLevel(hWnd);
	if (lRet != EC_OK) {
		WinGlideErrorMessageBox(NULL, "Could not set DirectDraw cooperative level.");

		goto initialization_failure;
	}

	//See if a FourCC surface is being used
	if (gbUseFourCCSurface == FALSE) {
		WinGlideErrorMessageBox(NULL, "When using off-screen surfaces, this version of WinGlide only supports surfaces defined by a FourCC.");

		goto initialization_failure;
	}
	
	//Create the off screen surface
	lRet = CreateOffScreenSurface(dwWidth, dwHeight, gbUseDoubleBufferedOverlay, gdwBackBufferCount,
		gbUseFourCCSurface, gFourCC, hWnd);
	if (lRet != EC_OK) {
		HRESULT hResult;

		//Look at the details of the error message
		switch (lRet) {
		case EC_COULD_NOT_CREATE_PRIMARY_SURFACE:
			WinGlideErrorMessageBox(NULL, "Could not create primary surface.");

			break;

		case EC_COULD_NOT_CREATE_OFF_SCREEN_SURFACE:
			//Get the DxDraw error code
			hResult = GetDxDrawErrorCode();

			//Display an error message
			DisplayCouldNotCreateOffScreenSurfaceErrorMessage(hResult);

			break;

		case EC_COULD_NOT_GET_BACK_BUFFER:
			WinGlideErrorMessageBox(NULL, "Could not get back buffer overlay surface.");

			break;

		default:
			//Show an error here just in case
			WinGlideErrorMessageBox(NULL, "Could not create off-screen surface.");
		}

		goto initialization_failure;
	}

	//Create the color key brush
	ghColorKeyBrush = CreateSolidBrush(gcrColorKey);
	if (ghColorKeyBrush == NULL) {
		WinGlideErrorMessageBox(NULL, "Could not create color key brush.");

		goto initialization_failure;
	}

	//Create the gamma correction table if gamma correction is enabled
	//Also create the gamma correction table if a FourCC surface is being used
	if ((gbEnableGammaCorrection == TRUE) || (gbUseFourCCSurface == TRUE)) {
		//Allocate memory for the gamma correction table
		bRet = AllocateGammaCorrectionTable();
		if (bRet == FALSE) {
			ErrorLoadingWinGlideMessageBox(NULL, "Could not create gamma correction table.");

			goto initialization_failure;
		}

		if (gbUseFourCCSurface == TRUE) {
			//Make sure the only supported FourCC YUY2 is being used
			//TODO: maybe change how this works later
			if (gFourCC != mmioFOURCC('Y', 'U', 'Y', '2')) {
				char szFourCC[5];
				char szMessage[512];

				//Display an error message
				szFourCC[0] = (char)(gFourCC);
				szFourCC[1] = (char)(gFourCC >> 8);
				szFourCC[2] = (char)(gFourCC >> 16);
				szFourCC[3] = (char)(gFourCC >> 24);
				szFourCC[4] = 0;

				wsprintf(szMessage, "The FourCC %s is not supported by this version of WinGlide.\nThis version of WinGlide only supports the FourCC YUY2.", szFourCC);

				ErrorLoadingWinGlideMessageBox(NULL, szMessage);

				goto initialization_failure;
			}

			//Calculate the color conversion table
			if (gbEnableGammaCorrection == TRUE) {
				//Calculate the color conversion table
				CalculateColorConversionTable(gFourCC, gRedGamma, gGreenGamma, gBlueGamma);
			}
			else {
				//Calculate the color conversion table
				CalculateColorConversionTable(gFourCC, 1.0f, 1.0f, 1.0f);
			}
		}
		else {
			//Calculate the gamma correction table
			CalculateGammaCorrectionTable(gRedGamma, gGreenGamma, gBlueGamma);
		}
	}

	//Subclass the window and take care of other modifications to the window
	bRet = CWGSubclassWindow(hWnd, &gpPrevWndProc, SubclassWndProc, WINGLIDE_MODE_OFF_SCREEN_SURFACE);
	if (bRet == FALSE) {
		ErrorLoadingWinGlideMessageBox(NULL, "Window subclassing error.");

		goto initialization_failure;
	}
	//Remember the window handle of the subclassed window
	ghSubclassedWnd = hWnd;

	return TRUE;

	//Jump here if initialization failed
initialization_failure:

	//Clean up anything that was allocated
	OssWGShutdown();

	return FALSE;
}

VOID OssWGShutdown(VOID) {
	//Unsubclass the window if it was subclassed and clean up other modifications to the window
	CWGUnsubclassWindowIfSubclassed();

	//The window is no longer subclassed
	ghSubclassedWnd = NULL;
	
	//Destroy the color key brush if it was created
	if (ghColorKeyBrush != NULL) {
		DeleteObject(ghColorKeyBrush);
		ghColorKeyBrush = NULL;
	}
	
	//Destroy the off screen surface if it was created
	FreeOffScreenSurface();

	//Shutdown DirectDraw
	DestroyDirectDraw();

	//Unload the DirectDraw library if it was loaded
	UnloadDirectDraw();

	//Free the gamma correction table
	FreeGammaCorrectionTable();

	return;
}

VOID OssWGCopy3DfxFrameBuffer(VOID) {
	FxBool fxbRet;
	GrLfbInfo_t grLFBInfo;
	BYTE *p3DfxLFBStart;
	LPVOID pSurface;
	DWORD dwSurfacePitch;
	BYTE *pOverlay;
	LONG lOverlayPitch;
	LONG lRet;
	BOOL bLockedOverlay;
	DWORD *pGammaTable;
	LONG lAlignedWidth;
	BOOL bRet;

	//Get the pointer to the gamma correction table
	if ((gbEnableGammaCorrection == TRUE) || (gbUseFourCCSurface == TRUE)) {
		pGammaTable = GetGammaCorrectionTablePointer();
		if (pGammaTable == NULL) {
			return;
		}
	}

	//Handle updating the gamma correction table if necessary
	bRet = CheckForNewGammaCorrectionValues(&gRedGamma, &gGreenGamma, &gBlueGamma);
	if (bRet == TRUE) {
		if (gbUseFourCCSurface == TRUE) {
			//Calculate the new color conversion table
			CalculateColorConversionTable(gFourCC, gRedGamma, gGreenGamma, gBlueGamma);
		}
		else {
			//Calculate the new gamma correction table
			CalculateGammaCorrectionTable(gRedGamma, gGreenGamma, gBlueGamma);
		}
	}

	//Lock the off screen surface
	lRet = LockOffScreenSurface(&pSurface, &dwSurfacePitch);
	if (lRet == EC_OK) {
		bLockedOverlay = TRUE;
		
		pOverlay = pSurface;
		lOverlayPitch = dwSurfacePitch;	
			
		//Make sure the error message is not displayed
		gbDisplayErrorMessage = FALSE;
		//Update the previous error code
		ghPreviousErrorCode = DD_OK;
	}
	else {
		HRESULT hResult;

		//Get the error code
		hResult = GetDxDrawErrorCode();

		//Display the error message
		DisplayCouldNotFlipOffScreenSurfaceErrorMessage(ghSubclassedWnd, hResult, FALSE);
	}

	if (bLockedOverlay == TRUE) {
		//Lock the 3Dfx LFB
		ZeroMemory(&grLFBInfo, sizeof(grLFBInfo));
		grLFBInfo.size = sizeof(grLFBInfo);
		fxbRet = _grLfbLock(GR_LFB_READ_ONLY, GR_BUFFER_BACKBUFFER, 0,
			GR_ORIGIN_ANY, FXFALSE, &grLFBInfo);
		if (fxbRet == FXFALSE) {
			//Could not lock the 3Dfx LFB
			
			//Unlock the overlay surface
			UnlockOverlay();
			
			return;
		}

		//Setup the pointer to the 3Dfx LFB
		if (gOriginLocation == GR_ORIGIN_LOWER_LEFT) {
			p3DfxLFBStart = (BYTE *)grLFBInfo.lfbPtr +
				(grLFBInfo.strideInBytes * (gdwScreenHeight - 1));
		}
		else {
			p3DfxLFBStart = (BYTE *)grLFBInfo.lfbPtr +
				(grLFBInfo.strideInBytes * (gdwGlideScreenHeight - 1));
		}
		
		//Get the aligned width to copy
		lAlignedWidth = ((gdwScreenWidth + 3) & ~3) >> 2;
		
		
		//See if a FourCC surface is being used
		if (gbUseFourCCSurface == TRUE) {
			//Copy the frame from the 3Dfx to the DIB
			if (gbUseMMX == TRUE) {
				MMXColorConvertPitchCopy8(p3DfxLFBStart, pOverlay,
					-(LONG)grLFBInfo.strideInBytes, lOverlayPitch,
					lAlignedWidth, gdwScreenHeight,
					pGammaTable);
			}
			else {
				ColorConvertPitchCopy8(p3DfxLFBStart, pOverlay,
					-(LONG)grLFBInfo.strideInBytes, lOverlayPitch,
					lAlignedWidth, gdwScreenHeight,
					pGammaTable);
			}
		}
		//See if gamma correction is enabled
		else if (gbEnableGammaCorrection == TRUE) {
			//Copy the frame from the 3Dfx to the DIB
			if (gbUseMMX == TRUE) {
				MMXGammaPitchCopy8(p3DfxLFBStart, pOverlay,
					-(LONG)grLFBInfo.strideInBytes, lOverlayPitch,
					lAlignedWidth, gdwScreenHeight,
					pGammaTable);
			}
			else {
				GammaPitchCopy8(p3DfxLFBStart, pOverlay,
					-(LONG)grLFBInfo.strideInBytes, lOverlayPitch,
					lAlignedWidth, gdwScreenHeight,
					pGammaTable);
			}
		}
		else {
			//Copy the frame from the 3Dfx to the DIB
			if (gbUseMMX == TRUE) {
				MMXPitchCopy8(p3DfxLFBStart, pOverlay,
					-(LONG)grLFBInfo.strideInBytes, lOverlayPitch,
					lAlignedWidth, gdwScreenHeight);
			}
			else {
				FPUPitchCopy8(p3DfxLFBStart, pOverlay,
					-(LONG)grLFBInfo.strideInBytes, lOverlayPitch,
					lAlignedWidth, gdwScreenHeight);
			}
		}	

		//Unlock the overlay surface
		UnlockOffScreenSurface();

		//Unlock the 3Dfx LFB
		fxbRet = _grLfbUnlock(GR_LFB_READ_ONLY, GR_BUFFER_BACKBUFFER);
		if (fxbRet == FXFALSE) {
			//Could not unlock the 3Dfx LFB
			return;
		}

		//See if double buffering is enabled
		//Double buffering isn't used right now so always flip the surface

		//Flip the off screen surface
		lRet = FlipOffScreenSurface();

		//Display an error message if the blt failed
		if (lRet == EC_OK) {
			//Make sure the error message is not displayed
			gbDisplayErrorMessage = FALSE;
			//Update the previous error code
			ghPreviousErrorCode = DD_OK;
		}
		else {
			HRESULT hResult;

			//Get the error code
			hResult = GetDxDrawErrorCode();

			//Display the error message
			DisplayCouldNotFlipOffScreenSurfaceErrorMessage(ghSubclassedWnd, hResult, FALSE);
		}
	}

	return;
}

static LRESULT CALLBACK SubclassWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam) {
	RECT rect;
	LRESULT lResult;
	static BOOL fNoSize;
	static RECT oldWindowRect;
	LONG lRet;

	switch (message) {
	case WM_SYSCOMMAND:
		//See if system menu options are enabled
		if (gbSystemMenuOptions == TRUE) {
			BOOL bRet;

			//Handle WinGlide system menu options
			bRet = HandleWinGlideSystemMenuOptions(hWnd, wParam, lParam);

			//Break if the message was handled
			if (bRet == TRUE) {
				break;
			}

			//If the message was not handled, pass it on to the previous window procedure
		}

		//Send the WM_SYSCOMMAND message to the previous window procedure
		return CallWindowProc(gpPrevWndProc, hWnd, message, wParam, lParam);

	case WM_ENTERSIZEMOVE:
		//Send the message to the previous window procedure
		lResult = CallWindowProc(gpPrevWndProc, hWnd, message, wParam, lParam);
		
		fNoSize = TRUE;

		//Save the old size of the window
		GetWindowRect(hWnd, &oldWindowRect);

		return lResult;

	case WM_EXITSIZEMOVE:
		//Send the message to the previous window procedure
		lResult = CallWindowProc(gpPrevWndProc, hWnd, message, wParam, lParam);
		
		fNoSize = FALSE;

		//See if the size of the window was changed
		GetWindowRect(hWnd, &rect);

		if (((rect.bottom - rect.top) != (oldWindowRect.bottom - oldWindowRect.top)) ||
			((rect.right - rect.left) != (oldWindowRect.right - oldWindowRect.left))) {

			//Display the overlay
		}

		return lResult;

	case WM_MOVE:
		//Send the message to the previous window procedure
		lResult = CallWindowProc(gpPrevWndProc, hWnd, message, wParam, lParam);

		return lResult;

	case WM_SIZE:
		//Send the message to the previous window procedure
		lResult = CallWindowProc(gpPrevWndProc, hWnd, message, wParam, lParam);
		
		if (fNoSize == TRUE) {
//Allow updating as the window is being resized
//			break;
		}
		
		//Hide the overlay if the window is iconic
		if (IsIconic(hWnd)) {			
			break;
		}

		//Display the overlay

		//Display the off screen surface
		lRet = FlipOffScreenSurface();
		if (lRet == EC_OK) {
			//Make sure the error message is not displayed
			gbDisplayErrorMessage = FALSE;
			//Update the previous error code
			ghPreviousErrorCode = DD_OK;
		}
		if (lRet != EC_OK) {
			HRESULT hResult;

			//Get the error code
			hResult = GetDxDrawErrorCode();

			//Display the error message
			DisplayCouldNotFlipOffScreenSurfaceErrorMessage(hWnd, hResult, TRUE);
		}

		return lResult;

	case WM_DISPLAYCHANGE:
		//Send the message to the previous window procedure
		lResult = CallWindowProc(gpPrevWndProc, hWnd, message, wParam, lParam);

		//Display the overlay if the window is not iconic
		if (IsIconic(hWnd)) {
			break;
		}
		//Display the overlay

		return lResult;

	case WM_DESTROY:
		//Send the message to the previous window procedure
		lResult = CallWindowProc(gpPrevWndProc, hWnd, message, wParam, lParam);

		//Unsubclass the window because it is being destroyed
		CWGUnsubclassWindowBecauseDestroyed();

		//The window is no longer subclassed
		ghSubclassedWnd = NULL;

		return lResult;

	default:
		return CallWindowProc(gpPrevWndProc, hWnd, message, wParam, lParam);
	}
	return 0;	
}

static VOID ConstructFlipOffScreenSurfaceFailedErrorMessage(HRESULT hResult) {
	//See what error code was returned
	switch (hResult) {
	case DDERR_GENERIC:
		lstrcpy(gszErrorMessage, "Generic error");
		break;
	case DDERR_HEIGHTALIGN:
		lstrcpy(gszErrorMessage, "Height alignment error");
		break;
	case DDERR_INVALIDOBJECT:
		lstrcpy(gszErrorMessage, "Invalid object");
		break;
	//Unsupported stretch factor
	case DDERR_INVALIDPARAMS:
	//Attempted to make overlay destination rectangle bigger than
	//the screen
	case DDERR_INVALIDRECT:
		lstrcpy(gszErrorMessage, "Invalid window size");
		break;
	case DDERR_INVALIDSURFACETYPE:
		lstrcpy(gszErrorMessage, "Invalid surface type");
		break;
	case DDERR_NOSTRETCHHW:
		lstrcpy(gszErrorMessage, "No stretch hardware");
		break;
	case DDERR_NOTAOVERLAYSURFACE:
		lstrcpy(gszErrorMessage, "Not an overlay surface");
		break;
	case DDERR_SURFACELOST:
		lstrcpy(gszErrorMessage, "Surface lost");
		break;
	case DDERR_UNSUPPORTED:
		lstrcpy(gszErrorMessage, "Unsupported");
		break;
	case DDERR_XALIGN:
		lstrcpy(gszErrorMessage, "Horizontal alignment error");
		break;
	case DDERR_INVALIDCLIPLIST:
		lstrcpy(gszErrorMessage, "Invalid clip list");
		break;
	case DDERR_NOALPHAHW:
		lstrcpy(gszErrorMessage, "No alpha acceleration hardware");
		break;
	case DDERR_NOBLTHW:
		lstrcpy(gszErrorMessage, "No blitter hardware");
		break;
	case DDERR_NOCLIPLIST:
		lstrcpy(gszErrorMessage, "No clip list is available");
		break;
	case DDERR_NODDROPSHW:
		lstrcpy(gszErrorMessage, "No DirectDraw raster operation hardware");
		break;
	case DDERR_NOMIRRORHW:
		lstrcpy(gszErrorMessage, "No mirroring hardware");
		break;
	case DDERR_NORASTEROPHW:
		lstrcpy(gszErrorMessage, "No raster operation hardware");
		break;
	case DDERR_NOROTATIONHW:
		lstrcpy(gszErrorMessage, "No rotation hardware");
		break;
	case DDERR_NOZBUFFERHW:
		lstrcpy(gszErrorMessage, "No z-buffer hardware");
		break;
	case DDERR_SURFACEBUSY:
		lstrcpy(gszErrorMessage, "Surface busy");
		break;
	default:
		lstrcpy(gszErrorMessage, "Error");
	}	

	return;
}

static VOID DisplayCouldNotFlipOffScreenSurfaceErrorMessage(HWND hWnd, HRESULT hResult, BOOL bRepaint) {
	HDC hDC;
	HGDIOBJ hPrevObj;
	RECT rect;

	//Just in case, make sure hWnd is not NULL
	if (hWnd == NULL) {
		return;
	}
	
	//Display an error message
	gbDisplayErrorMessage = TRUE;

	//Only construct and display an error message if it is different than the previous one
	//or if the window should be repainted
	if ((hResult != ghPreviousErrorCode) || (bRepaint == TRUE)) {
		//Update the previous error code
		ghPreviousErrorCode = hResult;

		//Construct the error message
		ConstructFlipOffScreenSurfaceFailedErrorMessage(hResult);

		//Update the window
		//GetDC
		hDC = GetDC(hWnd);
		hPrevObj = SelectObject(hDC, ghColorKeyBrush);
		
		//Paint the background with the color key brush
		GetClientRect(hWnd, &rect);
		PatBlt(hDC, rect.left, rect.top, rect.right, rect.bottom, PATCOPY);
		
		//See if an error message should be displayed
		if (gbDisplayErrorMessage == TRUE) {
			DrawText(hDC, gszErrorMessage, -1, &rect, DT_CENTER | DT_SINGLELINE | DT_VCENTER);
		}

		//ReleaseDC
		SelectObject(hDC, hPrevObj);
		ReleaseDC(hWnd, hDC);
	}

	return;
}

static VOID DisplayCouldNotCreateOffScreenSurfaceErrorMessage(HRESULT hResult) {
	char szMessage[256];

	//Put the default first line in the message
	lstrcpy(szMessage, "Could not create off-screen surface.\n");

	//See which error occurred
	switch (hResult) {
	case DDERR_INCOMPATIBLEPRIMARY:
		lstrcat(szMessage, "DDERR_INCOMPATIBLEPRIMARY\n");
		break;
	case DDERR_INVALIDCAPS:
		lstrcat(szMessage, "DDERR_INVALIDCAPS\n");
		break;
	case DDERR_INVALIDOBJECT:
		lstrcat(szMessage, "DDERR_INVALIDOBJECT\n");
		break;
	case DDERR_INVALIDPARAMS:
		lstrcat(szMessage, "DDERR_INVALIDPARAMS\n");
		break;
	case DDERR_INVALIDPIXELFORMAT:
		lstrcat(szMessage, "DDERR_INVALIDPIXELFORMAT\n");
		break;
	case DDERR_NOALPHAHW:
		lstrcat(szMessage, "DDERR_NOALPHAHW\n");
		break;
	case DDERR_NOCOOPERATIVELEVELSET:
		lstrcat(szMessage, "DDERR_NOCOOPERATIVELEVELSET\n");
		break;
	case DDERR_NODIRECTDRAWHW:
		lstrcat(szMessage, "DDERR_NODIRECTDRAWHW\n");
		break;
	case DDERR_NOEMULATION:
		lstrcat(szMessage, "DDERR_NOEMULATION\n");
		break;
	case DDERR_NOEXCLUSIVEMODE:
		lstrcat(szMessage, "DDERR_NOEXCLUSIVEMODE\n");
		break;
	case DDERR_NOFLIPHW:
		lstrcat(szMessage, "DDERR_NOFLIPHW\n");
		break;
	case DDERR_NOMIPMAPHW:
		lstrcat(szMessage, "DDERR_NOMIPMAPHW\n");
		break;
	case DDERR_NOOVERLAYHW:
		lstrcat(szMessage, "DDERR_NOOVERLAYHW\n");
		break;
	case DDERR_NOZBUFFERHW:
		lstrcat(szMessage, "DDERR_NOZBUFFERHW\n");
		break;
	case DDERR_OUTOFMEMORY:
		lstrcat(szMessage, "Not enough memory.\n");
		break;
	case DDERR_OUTOFVIDEOMEMORY:
		lstrcat(szMessage, "Not enough video memory.\n");
		break;
	case DDERR_PRIMARYSURFACEALREADYEXISTS:
		lstrcat(szMessage, "DDERR_PRIMARYSURFACEALREADYEXISTS\n");
		break;
	case DDERR_UNSUPPORTEDMODE:
		lstrcat(szMessage, "DDERR_UNSUPPORTEDMODE\n");
		break;
	default:
		;
	}

	//Display the error message
	WinGlideErrorMessageBox(NULL, szMessage);

	return;
}
